//package globalExamples;

import oracle.olapi.metadata.mdm.MdmAttribute;
import oracle.olapi.metadata.mdm.MdmDimension;
import oracle.olapi.metadata.mdm.MdmHierarchy;
import oracle.olapi.metadata.mdm.MdmLevel;
import oracle.olapi.metadata.mdm.MdmLevelHierarchy;
import oracle.olapi.metadata.mdm.MdmMeasure;
import oracle.olapi.metadata.mdm.MdmMeasureDimension;
import oracle.olapi.metadata.mdm.MdmMetadataProvider;
import oracle.olapi.metadata.mdm.MdmPrimaryDimension;
import oracle.olapi.metadata.mdm.MdmSchema;

import oracle.express.olapi.data.full.ExpressDataProvider;
import oracle.express.olapi.data.full.ExpressSpecifiedCursorManager;
import oracle.express.olapi.transaction.ExpressTransactionProvider;

import oracle.olapi.data.source.CursorManagerSpecification;
import oracle.olapi.data.source.DataProvider;
import oracle.olapi.data.source.Source;
import oracle.olapi.data.source.StringSource;
import oracle.olapi.data.source.SpecifiedCursorManager;

import oracle.olapi.data.cursor.Cursor;
import oracle.olapi.data.cursor.CursorManager;
import oracle.olapi.data.cursor.ValueCursor;
import oracle.olapi.data.cursor.IncorrectDataTypeException;

import oracle.olapi.transaction.Transaction;
import oracle.olapi.transaction.TransactionProvider;
import oracle.olapi.transaction.NotCommittableException;
import oracle.olapi.transaction.TransactionalObjectInvalidException;
import oracle.olapi.transaction.TransactionTypeUnavailableException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import java.lang.Math;
import java.sql.SQLException;

import java.io.PrintWriter;
import java.io.FileOutputStream;
import java.io.IOException;

/**
 * Establishes a connection to Oracle OLAP in an Oracle Database instance.
 * Gets the root schema, and gets the MdmMeasureDimension and
 * MdmPrimaryDimension objects from the schema.
 * Creates a hash map to store the MdmObject objects by name, and provides
 * methods to retrieve those objects.
 * Provides methods for creating a Cursor for a Source and for displaying
 * the values of a Cursor.
 * This class uses the CursorPrintWriter class.
 *
 * <P>
 * @author Oracle Corporation
 */
public class Context10g extends Object
{
  /**
   * The ExpressTransactionProvider for this session.
   */
  public ExpressTransactionProvider tp = null;
  /**
   * The ExpressDataProvider for this session.
   */
  public ExpressDataProvider dp = null;
  /**
   * The Oracle JDBC OracleConnection for this session.
   */
  public oracle.jdbc.OracleConnection conn = null;
  /**
   * The MdmMetadataProvider for this session.
   */
  public MdmMetadataProvider mp = null;
  /**
   * The CursorPrintWriter for this session.
   */
  public CursorPrintWriter cpw = null;
 /**
   * The Properties object for storing the command-line arguments and other
   * parameters to use when creating the connection to the database.
   */
  private Properties props = new Properties();
  /**
   * The HashMap to store the metadata objects and their names.
   */
  private HashMap m_hashMap = null;

  private List m_dimensionList = null;
  private List m_measureList = null;

  // Maps MdmObject objects to their names.
//  private HashMap nameMap = new HashMap();

  // Specifies whether to display progress messages.
  private boolean _displayProgress = false;

  private String _server;
  private String _user;

  /**
   * Creates a new Context10g object and specifies that the messages relating
   * the progress of making the connection to the database and of creating
   * the DataProvider, TransactionProvider, and the MetadataProvider are not
   * displayed.
   */
  public Context10g(String[] args)
  {
    this(args, false);
  }

  /**
   * Creates a new Context10g object.
   * If the displayProgress parameter is true, displays messages relating
   * the progress of making the connection to the database and of creating
   * the DataProvider, TransactionProvider, and the MetadataProvider.
   *
   * @param displayProgress A boolean that specifies whether to display the
   *                        messages relating the progress of making the
   *                        connection to the database and of creating the
   *                        DataProvider, TransactionProvider, and
   *                        MetadataProvider.
   */
   public Context10g(String[] args, boolean displayProgress)
   {
     _storeCommandLineProperties(args);
     _displayProgress = displayProgress;
     // Create a CursorPrintWriter
     cpw = new CursorPrintWriter();
     if (_displayProgress)
       println("Connecting to the database...");

     // Connect to the database
     _connect();
  }

  /**
   * Makes the connection to the Oracle OLAP server.
   */
  private void _connect()
  {
    if (_displayProgress)
      println("Loading the JDBC driver...");

    try
    {
      Class.forName("oracle.jdbc.driver.OracleDriver");
      if (_displayProgress)
        println("JDBC driver loaded.");
    }
    catch(Exception e)
    {
      println("Cannot not load the JDBC driver. " + e);
    }

    if (_displayProgress)
      println("Setting up connection properties...");

    // From the Properties object, get the command-line arguments that
    // specify the URL and the username and password to use in creating
    // the connection to the Oracle database.
    // The URL has the following information:
    //    An Oracle JDBC driver
    //    The server running the Oracle database
    //    The number of the port on which Oracle OLAP is listening
    //    The system identifier (SID) of the Oracle instance
    // When running any of the example programs, specify a URL, a username and
    // a password as command-line arguments in the following format:
    //   -url jdbc:oracle:thin:@serverName:portNumber:sid -user username
    //   -password password
    // In the URL specification, replace  "serverName" with the hostname of
    // the server on which the Orcle database is running.
    // Replace "portNumber" with the number of the TCP/IP listener port for
    // the database (which is 1521 by default).
    // Replace "sid" with the system identifier (SID) of the Oracle instance
    // to which you want to connect.
    // An example url is "jdbc:oracle:thin:@myOracleServer:1521:orcl".
    // The example programs use the Global schema. To run an example program
    // specify the username "global" and the password "global".
    // An example of the command-line arguments is the following:
    //  -url jdbc:oracle:thin:@myOracleServer:1521:orcl -user global
    //  -password global

    String url = props.getProperty("url");
    String user = props.getProperty("user");
    String password = props.getProperty("password");

    setServer(url);
    setUser(user);
    System.out.println("Connecting to " + getServer() + " as " +
                   getUser() + "...");

    
    if (_displayProgress)
      println("Getting the connection...");

    try
    {
      conn = (oracle.jdbc.OracleConnection)
      java.sql.DriverManager.getConnection(url, user, password);
      if (_displayProgress)
        println("Connection made.");
    }
    catch(SQLException e) {
      println("Connection attempt failed." + e);
    }

    _createProviders();
  }

  /**
   * Creates and stores the TransactionProvider, ExpressDataProvider, and the
   * MdmMetadataProvider.
   */
  private void _createProviders()
  {
    if (_displayProgress)
      println("Creating a TransactionProvider...");
    tp = new ExpressTransactionProvider();

    // Add another property to notify Oracle OLAP to use unique values,
    // which include the hierarchy, the level, and the local value for an
    // element of a dimension.
    props.setProperty("HierarchyValueType", "unique");

    if (_displayProgress)
      println("Creating a DataProvider...");
    dp = new ExpressDataProvider(conn, tp, props);
    // Initialize the DataProvider
    try
    {
    dp.initialize();
    }
    catch(SQLException e)
    {
      println("Cannot not initialize the DataProvider. " + e);
    }

    // Get an MdmMetadataProvider.
    if (_displayProgress)
      println("Getting the MDM MetadataProvider...");
    try
    {
      mp = (MdmMetadataProvider) dp.getDefaultMetadataProvider();
    }
    catch (Exception e)
    {
      println("Cannot create the MDM metadata provider." + e);
    }
  }

  /**
   * Gets the stored DataProvider.
   */
  public DataProvider getDataProvider()
  {
    return dp;
  }

  /**
   * Gets the stored MdmMetadataProvider.
   */
  public MdmMetadataProvider getMetadataProvider()
  {
    return mp;
  }

  /**
   * Gets the stored TransactionProvider.
   */
  public TransactionProvider getTransactionProvider()
  {
    return tp;
  }

  /**
   * From the top-level MdmSchema, gets the subschemas.
   * If the user is GLOBAL, then gets the GLOBAL_CAT subschema and gets
   * a list of the MdmDimension objects and the list of MdmMeasure objects
   * for that subschema.
   * If the user is not GLOBAL, then gets a list of MdmDimension objects and
   * a list of MdmMeasure objects from the root MdmSchema.
   * From the lists, the method creates a hash map to store the names of the
   * dimensions and measures.
   *
   * @param schema The top-level Oracle OLAP MdmSchema.
   */
  private HashMap loadHashMap (MdmSchema schema)
  {
    if (schema.getName().equals("ROOT"))
    {
      // Create an empty HashMap
      HashMap map = new HashMap();

      boolean isGlobal = false;
      MdmSchema schemaToUse = null;

      if (getUser().toUpperCase().equals("GLOBAL"))
      {
        isGlobal = true;

        List subSchemas = schema.getSubSchemas();
        Iterator subSchemasItr = subSchemas.iterator();

        while (subSchemasItr.hasNext())
        {
          schemaToUse = (MdmSchema) subSchemasItr.next();
          if (schemaToUse.getName().equals("GLOBAL_CAT"))
          {
            schema = schemaToUse;
            break;
          }
        }

      }

      // Get the dimension list
      m_dimensionList = schema.getDimensions();
      Iterator objIter = m_dimensionList.iterator();
      MdmPrimaryDimension mdmPDim = null;

      // For each dimension in the list...
      while(objIter.hasNext())
      {
        mdmPDim = (MdmPrimaryDimension) objIter.next();
        // Put its upper case name into the hash map as the key with the
        // object as the value.
        map.put(mdmPDim.getName().toUpperCase(), mdmPDim);
      }

      // Get the measure list
      if (isGlobal)
      {
        //m_measureList = schemaToUse.getMeasures();
        m_measureList = schema.getMeasures();
      }
      else
      {
        MdmMeasureDimension mDim = (MdmMeasureDimension)
                                    //schemaToUse.getMeasureDimension();
                                    schema.getMeasureDimension();
        m_measureList = mDim.getMeasures();
      }

      objIter = m_measureList.iterator();

      MdmMeasure mdmMeasure = null;
      // For each measure in the list...
      while(objIter.hasNext())
      {
        mdmMeasure = (MdmMeasure) objIter.next();
        // Put its upper case name into the hash map as the key with the
        // object as the value.
        map.put(mdmMeasure.getName().toUpperCase(), mdmMeasure);
      }

      return map;
    }
    else
      return null;
  }

  /**
   * Gets the MdmMeasure that has the specified name.
   *
   * @param name The name of the MdmMeasure that you want.
   *
   * @return The MdmMeasure that has the specified name or null.
   */
  public MdmMeasure getMdmMeasureByName(String name)
  {
    return getMdmMeasuresByName(new String[] {name})[0];
  }

  /**
   * Gets an array of MdmMeasure objects that has one MdmMeasure for each
   * specified name.
   * If the name is not in the hash map of MdmObject objects, the array
   * element for that name is null.
   *
   * @param names The names of the MdmMeasure objects that you want.
   *
   * @return An array of the MdmMeasure objects that have the specified names.
   */
  public MdmMeasure[] getMdmMeasuresByName(String[] names)
  {
    // If the hash map has not been created yet, create it.
    if (m_hashMap == null)
      m_hashMap = loadHashMap(mp.getRootSchema());

    // Create an array of nulls.
    MdmMeasure[] mdmMeasures = new MdmMeasure[names.length];
    // If the hash map load succeeded...
    if (m_hashMap != null)
    {
      // Loop over the names...
      for (int i=0; i<names.length; i++)
      {
        // Get the HashMap value for this name.
        String val = names[i].toUpperCase();
        mdmMeasures[i] = (MdmMeasure) m_hashMap.get(val);
      }
    }
    return (mdmMeasures);
  }

    /**
     * Gets the MdmPrimaryDimension that has the specified name.
     *
     * @param name The name of the MdmPrimaryDimension that you want.
     *
     * @return The MdmPrimaryDimension that has the specified name or null.
     */
    public MdmPrimaryDimension getMdmPrimaryDimensionByName(String name)
    {
      return getMdmPrimaryDimensionsByName(new String[] {name}) [0];
    }

  /**
   * Gets an array of MdmPrimaryDimension objects that has one
   * MdmPrimaryDimension for each specified name.
   * If the name is not in the hash map of MdmObject objects, the array
   * element for that name is null.
   *
   * @param names The names of the MdmPrimaryDimension objects that you want.
   *
   * @return An array of the the MdmPrimaryDimension objects that have the
   *         specified names.
   */
  public MdmPrimaryDimension[] getMdmPrimaryDimensionsByName(String[] names)
  {
    // If the hash map has not been created yet, create it.
    if (m_hashMap == null)
      m_hashMap = loadHashMap(mp.getRootSchema());

    // Create an array of nulls.
    MdmPrimaryDimension[] mdmPDims = new MdmPrimaryDimension[names.length];

    // If the hash map load succeeded...
    if (m_hashMap != null)
    {
      // Loop over the names
      for (int i=0; i<names.length; i++)
      {
		    // Make the name uppercase
  		  String val = names[i].toUpperCase();
        // Get the MdmPrimaryDimension with that name
  		  mdmPDims[i] = (MdmPrimaryDimension) m_hashMap.get(val);
  	  }
    }
  	return (mdmPDims);
  }

  public MdmHierarchy getHierarchyByName(MdmPrimaryDimension mdmPDim, String name)
  {
    List mdmHiers = mdmPDim.getHierarchies();
    Iterator mdmHierItr = mdmHiers.iterator();
    while(mdmHierItr.hasNext())
    {
      MdmHierarchy mdmHier = (MdmHierarchy) mdmHierItr.next();
      if (mdmHier.getName().equals(name))
        return mdmHier;
    }
    return null;
  }

  /**
   * Gets the MdmLevel that has the specified name.
   *
   * @param mdmLvlHier The MdmLevelHierarchy that contains the level that
   *                   you want.
   *
   * @param levelName A String that specifies the name of the level that
   *                  you want.
   *
   * @return The MdmLevel that has the specified name.
   */
  public MdmLevel getLevelByName(MdmLevelHierarchy mdmLvlHier, String levelName)
  {
    List mdmLevels = mdmLvlHier.getLevels();
    Iterator mdmLevelsItr = mdmLevels.iterator();
    while (mdmLevelsItr.hasNext())
    {
      MdmLevel mdmLevel = (MdmLevel) mdmLevelsItr.next();
      if ((mdmLevel.getName()).equals(levelName))
        return mdmLevel;
    }
    return null;
  }

  /**
   * Gets the Source for the specified level of a hierarchy.
   *
   * @param mdmLvlHier The MdmLevelHierarchy that contains the level for which
   *                   you want the Source.
   *
   * @param levelName A String that specifies the name of the level for
   *                  which you want the Source.
   *
   * @return The Source for the specified level.
   */
  public Source getLevelSource(MdmLevelHierarchy mdmLvlHier, String levelName)
  {
    return (getLevelByName((MdmLevelHierarchy)mdmLvlHier, levelName)).getSource();
  }

  /**
   * Displays the inputs of the specified Source.
   *
   * @param source The Source for which you want the inputs.
   */
  public void displayInputs(Source source)
  {
    Set inputs = source.getInputs();
    int numInputs = inputs.size();
    println("The number of inputs of " + source.getID() + "is "
             + numInputs + ".");
    println("The inputs are:");

    Iterator inputsItr = inputs.iterator();
    int i = 1;
    while (inputsItr.hasNext())
    {
      Source input = (Source) inputsItr.next();
      println(i + ": " + input.getID());
      i++;
    }
  }

  /**
   * Displays a line of text.
   *
   * @param line A String that contains the text that you want to display.
   */
  public void println(String line)
  {
    cpw.println(line);
  }

  /**
   * Prepares and commits the current Transaction.
   */
   public void commit()
   {
     try
     {
       tp.prepareCurrentTransaction();
     }
     catch(NotCommittableException e)
     {
       println("Cannot prepare the current Transaction. " + e);
     }
     tp.commitCurrentTransaction();
   }

  /**
   * Stores the command-line arguments used to run the example program.
   *
   * @param args A String array that contains the command-line arguments.
   */
   private void _storeCommandLineProperties(String[] args)
   {
     for (int i = 0; i < args.length; i += 2)
     {
	     if (i + 1 == args.length)
		   {
         throw new IllegalArgumentException("Command-line arguments must be " +
                   "specified in the form -<PropertyName> <PropertyValue>, " +
                   "with name and value separated by whitespace.");
       }
	     props.put(args[i].substring(1), args[i + 1]);
     }
   }

  /**
   * Creates a Cursor and prints the values to a file.
   *
   * @param source The Source for which to create a Cursor.
   *
   * @param path_filename The path and filename to which to send the output.
   *                      If you do not supply a path and filename, the output
   *                      goes to the screen.
   */
  public void printCursorToFile(Source source, String path_filename)
  {
    // Create a CursorPrintWriter that sends values to a file
    try
    {
      cpw = new CursorPrintWriter(new PrintWriter(
                                  new FileOutputStream(path_filename, true)));
    }
    catch (IOException e)
    {
      System.out.println("Error opening " + path_filename);
      System.exit(0);
    }

    createCursor(source, cpw);
  }

  /**
   * Creates a Cursor.
   *
   * @param source The Source for which to create a Cursor.
   * @param cpw The CursorPrintWriter to use for sending the output to
   *            the screen or to a file.
   */
  private void createCursor(Source source, CursorPrintWriter cpw)
  {
    CursorManagerSpecification curMngrSpec =
                        dp.createCursorManagerSpecification(source);
    SpecifiedCursorManager specCurMngr =
                        dp.createCursorManager(curMngrSpec);
    Cursor cursor = null;
    try
    {
      cursor = specCurMngr.createCursor();
    }
    catch (Exception e)
    {
      cpw.println("Cannot not create Cursor. Caught exception " + e);
    }

    //cpw.printCursor(cursor);
    cpw.printCursor(cursor, false);

    // Close the SpecifiedCursorManager.
    specCurMngr.close();

  }
  /**
   * Creates a Cursor for the specified Source and displays the values of
   * the Cursor, with unique values for the dimension elements.
   */
  public void displayResult(Source source)
  {
    _displayResult(source, false);
  }

  /**
   * Creates a Cursor for the specified Source and displays the values of
   * the Cursor, with unique values for the dimension elements if displayLocVal
   * is false and with local values only if displayLocVal is true.
   * This method retrieves local values only when the value separation String
   * is the default double colons (::).
   *
   * @param source The Source for which you want to create a Cursor and
   *               display its values.
   *
   * @param displayLocVal A boolean that specifies whether to display
   *                      unique or local dimension element values.
   */
  private void _displayResult(Source source, boolean displayLocVal)
  {
    CursorManagerSpecification cursorMngrSpec =
                       dp.createCursorManagerSpecification(source);
    SpecifiedCursorManager cursorManager =
                       dp.createCursorManager(cursorMngrSpec);
    Cursor cursor = cursorManager.createCursor();

    cpw.printCursor(cursor, displayLocVal);

    // Close the CursorManager.
    cursorManager.close();
  }


  /**
   * Displays the values of the specified Cursor.
   * This method displays the unique value of dimension elements.
   *
   * @param cursor The Cursor that has the values you want to display.
   */
  public void displayCursor(Cursor cursor)
  {
    _displayCursor(cursor, false);
  }

  /**
   * Displays the values of the specified Cursor.
   *
   * @param cursor The Cursor that has the values you want to display.
   *
   * @param displayLocVal A boolean that specifies whether to display
   *                      unique or local dimension element values.
   */
  public void displayCursor(Cursor cursor, boolean displayLocVal)
  {
    _displayCursor(cursor, displayLocVal);
  }

  /**
   * Displays the values of the specified Cursor.
   *
   * @param cursor The Cursor that has the values you want to display.
   *
   * @param displayLocVal A boolean that specifies whether to display
   *                      unique or local dimension element values.
   */
  private void _displayCursor (Cursor cursor, boolean displayLocVal)
  {
    cpw.printCursor(cursor, displayLocVal);
  }


  /**
   * Creates a Cursor for the specified Source and displays its values;
   * the values of any dimension elements are only the local values.
   *
   * @param source The Source for which you want to create a Cursor and
   *               display its values.
   */
  public void displayResultWithLocalValues(Source source)
  {
    _displayResult(source, true);
  }

  /**
   * Creates a Cursor for the Source produced by a TopBottomTemplate and
   * displays the values of the Cursor.
   *
   * @param source The Source returned by the getSource method of a
   *               DynamicDefinition for the TopBottomTemplate example.
   */
  public void displayTopBottomResult(Source source)
  {
    CursorManagerSpecification cursorMngrSpec =
                       dp.createCursorManagerSpecification(source);
    SpecifiedCursorManager cursorManager =
                       dp.createCursorManager(cursorMngrSpec);
    Cursor cursor = cursorManager.createCursor();

    cpw.displayTopBottomResult(cursor);

    // Close the CursorManager.
    cursorManager.close();
  }

  /**
   * Display the values of a Cursor in a crosstab format
   */
  public void displayCursorAsCrosstab(Cursor cursor)
  {
    cpw.printAsCrosstab(cursor);
  }

  public String getLocalValue(String uniqueValue)
  {
    int index = 0;
    if (uniqueValue.indexOf("::") > 0)
    {
      index = uniqueValue.lastIndexOf("::");
      String localValue = uniqueValue.substring((uniqueValue.lastIndexOf("::")
                                                  + 2),
                                                 uniqueValue.length());
      return(localValue);
    }
    else
      return uniqueValue;
  }

   /**
    * Specify the name of the Oracle OLAP server for the connection.
    */
   public void setServer(String url)
   {
     int indexAtSign = url.indexOf("@");
     int indexColonAfterServer = url.indexOf(":", indexAtSign);
     String server = url.substring(indexAtSign + 1, indexColonAfterServer);
     _server = server;
   }

   /**
    * Get the name of the Oracle OLAP server for the connection.
    */
   public String getServer()
   {
     return _server;
   }

   /**
    * Specify the username for the connection.
    */
   public void setUser(String user)
   {
     _user = user;
   }

   /**
    * Get the username for the connection.
    */
   public String getUser()
   {
     return _user;
   }
}
